(in-package "CL-USER")

;desc represents a machine description.
(defstruct desc
  (consts nil) ;constants
  (functs nil) ;functions
  (defs nil)   ;definitions
  (vars nil)   ;variables
  (init nil)   ;initialization formula
  (trans nil)  ;transition formula
  (spec nil))  ;specification formula

(deftype form-type () '(or (cons (member mem) (cons integer (cons integer null)))
			   (cons (member bv)  (cons integer null))
			   (cons (member int) (cons integer null))
			   (cons (member mv) list)))

;formula gives our dag representation of formulas.
(defstruct (formula (:PRINT-FUNCTION print-form))
  (fn nil :type symbol)           ;specifies the function (or type of atom) is represented by the formula
  (type '(bv 1) :type form-type) ;; can be int, bv, mv, or mem
  (value 0 :type fixnum) ; when simplifying, each unique formula will be given a number.
  (args nil :type list)           ;the arguments to the function (or info about the atom)
  (depth 0 :type fixnum)
  (slot1 nil)                     ;used as a temporary holding slot for some algorithms
  (slot2 nil))                    ;used as a temporary holding slot for some algorithms

;this is a struct for representing user-defined functions.
(defstruct funct
  (name nil :type symbol)    ;name of the function
  (type '(bv 1))    ;type of the function
  (params nil :type list)  ;parameter list for the function
  (body nil))   ;body of the function. 


(defvar *fhash* (make-hash-table :test 'equal :rehash-size 1.5 :rehash-threshold 0.7))

(defvar *ftrie* nil)

(declaim (type fixnum *fc*))
(defvar *fc* 0)

(defvar *ap* t)
(defvar *up* t)

(defun fix-form (x)
  (if (formula-p x)
      (formula-value x)
    x))

(declaim (ftype (function (symbol form-type list) list)))
(defun form-key (fn tp args)
  (declare (type symbol fn) (type form-type tp) (type list args))
  (cons (mapcar #'fix-form args) (cons fn tp)))

(declaim (ftype (function (list fixnum) fixnum)))
(defun max-depth (lst max)
  (declare (type list lst) (type fixnum max))
  (cond ((endp lst) max)
	((formula-p (car lst))
	 (max-depth (cdr lst) (max max (formula-depth (car lst)))))
	(t (max-depth (cdr lst) max))))

(declaim (ftype (function (&key (:fn symbol) (:type form-type) (:args list)) formula) make-unique-formula))
(defun make-unique-formula (&key (fn 'none) (type '(bv 1)) (args nil))
  (declare (type symbol fn) (type form-type type) (type list args))
  (if *up*
      (let* ((key (form-key fn type args))
	     ;; (my-balance (cons fn (mapcar #'fix-form args))))
	     (uf (gethash key *fhash*)))
	(if uf
	    uf
	  (let ((nf (make-formula :fn fn 
				  :type type
				  :depth (1+ (max-depth args 0))
				  :value (if (eq fn 'not) 
					     (1- (formula-value (car args)))
					   (decf *fc* 2))
				  :args args)))
	    ;; (when (= (formula-value nf) -97) (break))
	    (setf (gethash key *fhash*) 
		  nf))))
    (make-formula :fn fn
		  :type type
		  :depth (1+ (max-depth args 0))
		  :value (if (eq fn 'not)
			     (1- (formula-value (car args)))
			   (decf *fc* 2))
		  :args args)))

;the single bit 0 form. we create one to avoid duplicates.
(declaim (type formula *zero*))
(defvar *zero* (make-unique-formula :fn 'const :args '(0)))

;the single bit 1 form. we create one to avoid duplicates.
(declaim (type formula *one*))
(defvar *one* (make-unique-formula :fn 'const :args '(1)))

; the single junk bit.
(declaim (type formula *junk*))
(defvar *junk* (make-unique-formula :fn 'junk :args nil))

(defun clistp (list)
  (cond ((endp list) t)
	((eq (formula-fn (car list)) 'const) (clistp (cdr list)))
	(t nil)))

(defun const-to-str (const)
  (format nil "0b~{~:[1~;0~]~}"
	  (mapcar (lambda (x) (eq x *zero*)) (formula-args const))))

(defun const-to-symbol (const)
  (read-from-string (const-to-str const)))

(declaim (ftype (function (formula stream t) null) print-form))
(defun print-form (s stream depth)
  (declare (ignore depth))
  (let ((f (format nil "~a:~a:~a" 
		   (formula-fn s)
		   (formula-type s)
		   (formula-value s))))
    (format stream "~a" (cons f (formula-args s)))))

;; bit-vectors. we put these in a struct because lisp is too stupid to
;; know the difference between arrays of formulas and arrays of arrays
;; of formulas.

(defstruct (vec (:PRINT-FUNCTION print-vec))
  (array (make-array 1 :element-type 'formula :initial-element *junk*) :type (simple-array formula))
  (value 0 :type fixnum))

(declaim (type vec *junk-vec*))
(defvar *junk-vec* (make-vec))

(declaim (ftype (function (fixnum) vec) new-vec))
(defun new-vec (size)
  (declare (type fixnum size))
;;   (when (= size 1)
;;     (break))
    (make-vec :array (make-array size
				 :element-type 'formula
				 :initial-element *junk*)))
	
(defun vec-key (vec)
  (mapcar #'fix-form (coerce (vec-array vec) 'list)))

(declaim (ftype (function (vec) vec) make-unique-vec))
(defun make-unique-vec (vec)
  (declare (type vec vec))
  (let ((key (vec-key vec)))
    (multiple-value-bind
	(uv foundp)
	(gethash key *fhash*)
	(cond (foundp uv)
	      (t (setf (vec-value vec) (decf *fc* 2))
		 (setf (gethash key *fhash*) vec))))))

(declaim (ftype (function (vec fixnum) formula) vec-get-bit))
(defun vec-get-bit (vec bit) (the formula (aref (vec-array vec) bit)))

(declaim (ftype (function (vec fixnum formula) formula) vec-set-bit))
(defun vec-set-bit (vec bit form) (setf (aref (vec-array vec) bit) form))

;; (defmacro vec-bit (vec bit)
;;   `(the formula (aref (vec-array ,vec) ,bit)))

(declaim (ftype (function (form-vec) fixnum) vec-num-bits))
(defun vec-num-bits (vf)
  (declare (type form-vec vf))
  (if (vec-p vf) (array-dimension (vec-array vf) 0) 1))
		
(declaim (ftype (function (vec stream t) null) print-vec))
;; (defun print-vec (s stream depth)
;;   (declare (ignore depth))
;;   (format stream "(vec:~D ~A)" (vec-value s) (vec-array s)))

(defun print-vec (s stream depth)
  (declare (ignore depth))
  (if (dotimes (i (vec-num-bits s) t)
	(let ((si (vec-get-bit s i)))
	  (unless (or (eq si *one*)
		      (eq si *zero*))
	    (return nil))))
      (progn (format stream "0b")
	     (loop for i from (1- (vec-num-bits s)) downto 0
		   do (format stream "~:[1~;0~]" (eq (vec-get-bit s i) *zero*))))
    (format stream "(vec:~D ~A)" (vec-value s) (vec-array s))))

(deftype form-vec () '(or formula vec))

;; memories.

(defstruct mem
  (array (make-array 1
		     :element-type 'vec
		     :initial-element *junk-vec*)
	 :type (simple-array vec)))

(declaim (ftype (function (mem) fixnum) mem-num-words))
(defun mem-num-words (mem) (array-dimension (mem-array mem) 0))

(declaim (ftype (function (mem) fixnum) mem-wordsize))
(defun mem-wordsize (mem) (vec-num-bits (the vec (aref (mem-array mem) 0))))

(declaim (ftype (function (mem fixnum) form-vec) mem-get-word))
(defun mem-get-word (mem i)
  (let ((vec (the vec (aref (mem-array mem) i))))
    (if (= (vec-num-bits vec) 1)
	(vec-get-bit vec 0)
      vec)))

(declaim (ftype (function (mem fixnum form-vec) form-vec) mem-set-word))
(defun mem-set-word (mem i fv)
  (cond ((formula-p fv)
	 (let ((vec (new-vec 1)))
	   (vec-set-bit vec 0 fv)
	   (setf (aref (mem-array mem) i) vec)
	   fv))
	(t
	 (setf (aref (mem-array mem) i) fv))))

(declaim (ftype (function (mem fixnum fixnum) formula) mem-get-bit))
(defun mem-get-bit (mem i j)
  (the formula (aref (vec-array (the vec (aref (mem-array mem) i))) j)))

(declaim (ftype (function (mem fixnum fixnum formula) formula) mem-set-bit))
(defun mem-set-bit (mem i j form)
  (setf (aref (vec-array (aref (mem-array mem) i)) j) form))

(declaim (ftype (function (fixnum fixnum) mem) new-mem))
(defun new-mem (nw ws)
  (declare (type fixnum nw) (type fixnum ws))
  (let ((mem (make-mem :array (make-array nw
					  :element-type 'vec
					  :initial-element (make-vec)))))
    (dotimes (i nw mem) (mem-set-word mem i (new-vec ws)))))

(deftype vec-mem () '(or vec mem))
(deftype form-vec-mem () '(or formula vec mem))
